/********************************************************************
*
*  LPC15xx - RC5 encoder (transmitter)
*
*  Use SCT0 SCT0_OUT0 @ P0_11 to send modulated (36 KHz carrier) RC5 frames
*  Reference: AN10210 and AN10722
*  RC5 format:
*
*      | S | F | C |   5 sytem bits    |    6 command bits     |
*      | 1 | 1 | 0 | 0 | 0 | 0 | 0 | 1 | 1 | 0 | 1 | 1 | 1 | 1 |
*
*        +-+ +---+ +-+ +-+ +-+ +-+   +-+ +---+   +-+ +-+ +-+ +-+
*        | | |   | | | | | | | | |   | | |   |   | | | | | | | |
*   -----+ +-+   +-+ +-+ +-+ +-+ +---+ +-+   +---+ +-+ +-+ +-+ +----
*
*        | |
*        | +--------------------------------------------------+
*        |                                                    |
*        +-+     +-+     +-+                  +-+     +-+
*        | |     | |     | |      32 x        | |     | |
*   -----+ +-----+ +-----+ +- - - - - - - - - + +-----+ +-------
*
*        |   CP  |
*
*  CP = 27.777 usec (= 36 KHz)
*  25% duty cycle: high time = 6.944 usec
*  NOTE: SIGNAL IS INVERTED BECAUSE INVERTING TRANSISTOR AT MC100 BOARD
*
********************************************************************/
#include "LPC15xx.h"                                  // LPC15xx definitions

#define RC5_DATA0()       LPC_IOCON->PIO0_30 = 0x88   // input low (pull down)
#define RC5_DATA1()       LPC_IOCON->PIO0_30 = 0x90   // input high (pull up)

static uint8_t  state;                                 // driver state
static uint8_t  count;                                 // bit counter
static uint8_t  flag;                                  // frame send ready flag
static uint16_t frame;                                 // frame

void RC5_Send(uint16_t fr)
{
    frame = fr << 2;                                   // MSB first, 14 bits shift left 2
    state = 0;
    flag  = 0;                                         // clear ready flag
    count = 14;                                        // set bit count to 14

    LPC_SCT0->EVEN = (1 << 4);                         // event 4 generates interrupt

    while (!flag) ;                                    // wait for ready flag
}

void SCT0_IRQHandler(void)
{
    if (count)                                         // send next bit if required
    {
        if (state)
        {
            if (frame & 0x8000)
            {
                RC5_DATA1();
            }
            else
            {
                RC5_DATA0();
            }
            state = 0;
            count --;
            frame <<= 1;
        }
        else
        {
            if (frame & 0x8000)
            {
                RC5_DATA0();
            }
            else
            {
                RC5_DATA1();
            }
            state = 1;
        }
    }
    else
    {
        RC5_DATA0();                                       // inactive carrier pulse
        LPC_SCT0->EVEN = 0;                                // disable interrupt
        flag = 1;                                          // signal ready
    }
    LPC_SCT0->EVFLAG = (1 << 4);                           // clear event 4 flag
}

void RC5_Init(void)
{
    RC5_DATA0();                                           // SCT0_IN0 low
    LPC_SYSCON->SYSAHBCLKCTRL1 |= (1 << 2);                // enable the SCT0 clock
    LPC_PMUX->SCT0_P_MUX0 = 3;                             // SCT0_IN0 at P0_30
    LPC_SCT0->OUTPUT      = (1 << 0);                      // SCT0_OUT0 (IR output) is high
    LPC_SWM->PINASSIGN7  |= 0x0000FF00;                    // ASSIGN7(15:8) = FF
    LPC_SWM->PINASSIGN7  &= 0xFFFF0BFF;                    // P0_11 is SCT0_OUT0, ASSIGN7(15:8)

/********************************************************************
*  SCT_L: low part configuration:
********************************************************************/
                                              
    LPC_SCT0->MATCH[0].L    = (SystemCoreClock / 48000)-1; // 75% of 36 KHz
    LPC_SCT0->MATCHREL[0].L = (SystemCoreClock / 48000)-1;
    LPC_SCT0->MATCH[1].L    = (SystemCoreClock / 36000)-1; // 36 KHz
    LPC_SCT0->MATCHREL[1].L = (SystemCoreClock / 36000)-1;

    LPC_SCT0->EVENT[0].STATE = 0x00000003;                 // event 0 happens in all states
    LPC_SCT0->EVENT[0].CTRL  = (0 << 0)  |                 // MATCHSEL [3:0]   = related to match 0
                               (0 << 4)  |                 // HEVENT   [4]     = use L state & match
                               (0 << 5)  |                 // OUTSEL   [5]     = select input from IOSEL
                               (0 << 6)  |                 // IOSEL    [9:6]   = select input 0
                               (3 << 10) |                 // IOCOND   [11:10] = high level
                               (3 << 12) |                 // COMBMODE [13:12] = match AND IO condition
                               (0 << 14) |                 // STATELD  [14]    = STATEV is added to state
                               (0 << 15) |                 // STATEV   [19:15] = no state change
                               (0 << 20) |                 // MATCHMEM [20]    = "equal" to match
                               (0 << 21);                  // DIRECTION[22:21] = direction independent

    LPC_SCT0->EVENT[1].STATE = 0x00000003;                 // event 1 happens in all states
    LPC_SCT0->EVENT[1].CTRL  = (1 << 0)  |                 // MATCHSEL [3:0]   = related to match 1
                               (1 << 12) |                 // COMBMODE [13:12] = match condition only
                               (0 << 14) |                 // STATELD  [14]    = STATEV is added to state
                               (0 << 15);                  // STATEV   [19:15] = no state change

    LPC_SCT0->OUT[0].CLR     = (1 << 0);                   // IR LED low  @ event 0
    LPC_SCT0->OUT[0].SET     = (1 << 1);                   // IR LED high @ event 1
    LPC_SCT0->LIMIT_L        = (1 << 1);                   // events 1 is used as counter limit

/********************************************************************
*  SCT_H: high part configuration:
********************************************************************/

    LPC_SCT0->MATCH[0].H    = (SystemCoreClock / 1125)-1;  // 36 KHz / 32 ticks = 1125
    LPC_SCT0->MATCHREL[0].H = (SystemCoreClock / 1125)-1;

    LPC_SCT0->EVENT[4].STATE = 0x00000003;                 // event 4 only happens in all states
    LPC_SCT0->EVENT[4].CTRL  = (0 << 0)  |                 // MATCHSEL [3:0]   = related to match_H 0
                               (1 << 4)  |                 // HEVENT   [4]     = use H state & match
                               (1 << 12) |                 // COMBMODE [13:12] = match condition only
                               (0 << 14) |                 // STATELD  [14]    = STATEV is added to state
                               (0 << 15);                  // STATEV   [19:15] = no state change

    LPC_SCT0->LIMIT_H        = (1 << 4);                   // event 4 is used as H counter limit

    NVIC_EnableIRQ(SCT0_IRQn);                             // enable SCT interrupt

    LPC_SCT0->CTRL_U &= ~((1 << 2) | (1 << 18));           // unhalt both low and high counter
}
